#!/usr/bin/env ts-node

// Create a MetaTx from the CLI
//
// Usage:
// ts-node createMetaTxCLI.ts <function> <function_args> <contract_address> <forwarder_addr> <forwarder_chainid> <signer_sk>

import { argv } from 'process'
import { Wallet } from 'ethers'
import { createMetaTx } from './createMetaTx'
import {
  createTxDataForSetIdentity,
  createTxDataForSetTokenTypeRule,
  createTxDataForSetTransferRule,
  createTxDataForMint,
  createTxDataForGrantRole,
} from './createTxData'

function createTxDataForFunction(functionName: string, functionArgs: string[]) {
  switch (functionName) {
    case 'setIdentity':
      return createTxDataForSetIdentity({
        destination: functionArgs[0],
        identityInfo: {
          regions: functionArgs[1].split('/').map(Number),
          accreditationType: parseInt(functionArgs[2]),
          lastAmlKycChangeTimestamp: parseInt(functionArgs[3]),
          lastAccreditationChangeTimestamp: parseInt(functionArgs[4]),
          amlKycPassed: functionArgs[5] === 'true',
        },
      })
    case 'setTokenTypeRule':
      return createTxDataForSetTokenTypeRule({
        region: parseInt(functionArgs[0]),
        accreditation: parseInt(functionArgs[1]),
        tokenType: parseInt(functionArgs[2]),
        requiresAmlKyc: functionArgs[3] === 'true',
        isActive: functionArgs[4] === 'true',
      })
    case 'setTransferRule':
      return createTxDataForSetTransferRule({
        tokenType: parseInt(functionArgs[0]),
        recipientRegion: parseInt(functionArgs[1]),
        recipientAccreditation: parseInt(functionArgs[2]),
        transferRule: {
          lockDurationSeconds: parseInt(functionArgs[3]),
          requiresAmlKyc: functionArgs[4] === 'true',
          isActive: functionArgs[5] === 'true',
        },
      })
    case 'mint':
      return createTxDataForMint({
        to: functionArgs[0],
        value: parseInt(functionArgs[1]),
      })
    case 'grantRole':
      return createTxDataForGrantRole({
        to: functionArgs[0],
        role: parseInt(functionArgs[1]),
      })
    default:
      throw new Error(`Unsupported function: ${functionName}`)
  }
}

async function main() {
  const functionName = argv[2]
  const functionArgs = argv[3].split(',')
  const contractAddress = argv[4]
  const forwarderAddr = argv[5]
  const forwarderChainId = parseInt(argv[6])
  const signerSk = argv[7]
  const signer = new Wallet(signerSk)

  const metaTxWithSig = await createMetaTx(
    {
      from: signer.address,
      to: contractAddress,
      value: 0,
      gas: 1000000,
      nonce: new Date().getTime(),
      deadline: 0,
      data: createTxDataForFunction(functionName, functionArgs),
      signature: '',
    },
    forwarderAddr,
    forwarderChainId,
    signer,
  )

  console.log(metaTxWithSig)
}

main()
